/* Copyright (c) 2010 Nordic Semiconductor. All Rights Reserved.
 *
 * The information contained herein is property of Nordic Semiconductor ASA.
 * Terms and conditions of usage are described in detail in NORDIC
 * SEMICONDUCTOR STANDARD SOFTWARE LICENSE AGREEMENT. 
 *
 * Licensees are granted free, non-transferable use of the information. NO
 * WARRANTY of ANY KIND is provided. This heading must NOT be removed from
 * the file.
 */ 

/** @file
 * @brief Implementation of lib_ir_transmit
 */

#ifndef LIB_IR_TRANSMIT_H__
#define LIB_IR_TRANSMIT_H__

#include <stdint.h>
#include <stdbool.h>
#include "lib_ir_common.h"

/** Define either PWM0 or PWM1 to select the correct output
 */
#ifdef PWM0
  #define LIB_IR_PWM_OUTPUT(val) do{ PWMDC0 = val;      \
                                 }while(0);                               
#else 
  #ifdef PWM1
    #define LIB_IR_PWM_OUTPUT(val) do{ PWMDC1 = val;      \
                                   }while(0);
  #endif
#endif

/** @name   MISC FUNCTIONS  */
//@{
/* Misc function prototypes */

/** Function to initialize the IR transmit library.
 *  This function sets default parameters and initializes the PWM
 *  @param timer_period The period of the timer that is used in s 
 */
void lib_ir_transmit_init(uint8_t timer_period);

/** Function that should be called every timer interrupt
 *  This function increment counters and keeps track of the timing when sending the IR data
 */
void lib_ir_transmit_timer_irq_function(void);

/** Function that is called when a transmission is complete
 *  This function must be implemented outside the library
 */
void lib_ir_transmit_data_sent_callback(void);

/** @name   OPERATION FUNCTIONS  */
//@{
/* Operation function prototypes */

/** Function to start transmission of a RC-5 command.
 *  @param address The address of the receiver.    
 *  @param command The data payload sent to the address.
 *  @param repeat State if this is a repeated command or not, it will affect the toggle bit
 *  @return The status of the command, TRUE if success, FALSE if fail
 */
bool lib_ir_transmit_rc5_command(uint8_t address, uint8_t command, bool repeat);

/** Function to start transmission of a NEC command.
 *  @param address The address of the receiver.    
 *  @param inv_address The inverted address of the receiver.
 *  @param command The data payload sent to the address.
 *  @param inv_command The inverted data payload sent to the address.
 *  @param repeat State if this is a repeated command or not
 *  @return The status of the command, TRUE if success, FALSE if fail
 */
bool lib_ir_transmit_nec_command(uint8_t address, uint8_t inv_address, uint8_t command, uint8_t inv_command, bool repeat);

/** Function to start transmission of a SIRC command.
 *  @param address The address of the receiver.    
 *  @param command The data payload sent to the address.
 *  @param extended The extended word of the 20 bit version
 *  @param version The version of the protocol, 12-bit, 15-bit or 20-bit
 *  @param repeat State if this is a repeated command or not
 *  @return The status of the command, TRUE if success, FALSE if fail
 */
bool lib_ir_transmit_sirc_command(uint8_t address, uint8_t command, uint8_t extended, lib_ir_protocol_name_t version, bool repeat);

#endif
